from abc import ABC, abstractmethod
from typing import Callable, Any, List, Dict, Optional, Union, Tuple

class BaseOptimizer(ABC):
    def __init__(
        self,
        space: Optional[Union[Dict[str, Any], Callable]] = None,  
        metric: Optional[Union[str, List[str]]] = None,  
        mode: Optional[Union[str, List[str]]] = None,  
        objective_fn: Optional[Callable[[Dict[str, Any]], Any]] = None, 
        seed: int = 42, 
        points_to_evaluate: Optional[List[Dict[str, Any]]] = None,  
        evaluated_rewards: Optional[List[Union[float, List[float]]]] = None, 
        max_trials: Optional[int] = None,  
        max_time: Optional[float] = None, 
        early_stopping_fn: Optional[Callable[[List[Tuple[Dict[str, Any], Any]]], bool]] = None,
        verbose: bool = False,
    ):
        """
        Abstract base class for all optimizers (e.g., Optuna, Hyperopt, etc).

        This class defines the common interface and properties for hyperparameter
        optimization algorithms. Subclasses must implement the `optimize` and
        `get_best_config` methods.
        """
        self.space = space
        self.metric = metric or "score" 
        self.mode = mode or "min"
        self.objective_fn = objective_fn  
        self.seed = seed 
        self.points_to_evaluate = points_to_evaluate or [] 
        self.evaluated_rewards = evaluated_rewards or []  
        self.max_trials = max_trials
        self.max_time = max_time  
        self.early_stopping_fn = early_stopping_fn 
        self.trial_history = []
        self.elapsed_time: Optional[float] = None
        self.verbose = verbose

    @abstractmethod
    def optimize(self) -> List[Tuple[Dict[str, Any], Any]]:
        """
        Perform the optimization process.

        Returns:
            A list of tuples, where each tuple contains a configuration, score, and elapsed time.
        """
        pass

    @abstractmethod
    def get_best_config(self, include_score: bool = False) -> Union[Dict[str, Any], Dict[str, Any]]:
        """
        Retrieve the best configuration found during optimization.

        Returns:
            A dictionary representing the best hyperparameter configuration.
        """
        pass
